<?php

use Adianti\Registry\TSession;

/**
 * Classe para gerenciar criptografia de IDs usando chave única por sessão
 * 
 * Esta classe garante que cada usuário tenha uma chave única de criptografia
 * armazenada na sessão, tornando os IDs encriptados únicos e variáveis,
 * impedindo acesso não autorizado de outros usuários.
 * 
 * IMPORTANTE: Esta classe NÃO substitui validações de ownership/authorization.
 * Sempre valide que o usuário tem permissão para acessar o recurso (ex: WHERE id=:id AND user_id=:user_id).
 * Proteja contra CSRF e XSS em todos os endpoints.
 * 
 * NOTA: Chaves rotacionam por sessão. Links compartilhados ou abertos em outra sessão/navegador
 * não funcionarão após expiração da sessão original. Isso é intencional para segurança.
 */
class SessionIdEncryption
{
    private static $sessionKeyName = 'session_encryption_key';
    
    /**
     * Converte base64 padrão para base64url (URL-safe)
     * 
     * @param string $data Dados em base64
     * @return string Dados em base64url
     */
    private static function base64urlEncode($data)
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    /**
     * Converte base64url para base64 padrão
     * 
     * @param string $data Dados em base64url
     * @return string|false Dados em base64 ou false em caso de erro
     */
    private static function base64urlDecode($data)
    {
        // Adiciona padding se necessário
        $padding = strlen($data) % 4;
        if ($padding) {
            $data .= str_repeat('=', 4 - $padding);
        }
        return base64_decode(strtr($data, '-_', '+/'), true);
    }
    
    /**
     * Inicializa ou recupera a chave de criptografia da sessão
     * A chave é gerada uma vez por sessão e reutilizada
     * 
     * @return string Chave de criptografia da sessão (32 bytes em base64)
     */
    private static function getSessionKey()
    {
        $key = TSession::getValue(self::$sessionKeyName);
        
        if (empty($key)) {
            // Gera uma chave aleatória de 32 bytes (256 bits) e armazena em base64
            // Mais eficiente que bin2hex (armazena 44 chars vs 64 chars)
            $keyBytes = random_bytes(32);
            $key = base64_encode($keyBytes);
            TSession::setValue(self::$sessionKeyName, $key);
        }
        
        // Decodifica a chave para bytes
        return base64_decode($key, true);
    }
    
    /**
     * Encripta um ID usando a chave da sessão
     * 
     * @param mixed $id ID a ser encriptado (string ou número)
     * @return string ID encriptado em base64url (URL-safe)
     * @throws Exception
     */
    public static function encryptId($id)
    {
        if (empty($id) && $id !== 0 && $id !== '0') {
            return '';
        }
        
        try {
            // Converte o ID para string
            $idString = (string)$id;
            
            // Obtém a chave da sessão (já em bytes)
            $sessionKeyBytes = self::getSessionKey();
            
            // Verifica se a extensão Sodium está disponível
            if (function_exists('sodium_crypto_secretbox')) {
                // Deriva uma chave de 32 bytes a partir da chave da sessão
                $key = sodium_crypto_generichash($sessionKeyBytes, '', SODIUM_CRYPTO_SECRETBOX_KEYBYTES);
                $nonce = random_bytes(SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
                $encrypted = sodium_crypto_secretbox($idString, $nonce, $key);
                
                // Retorna em base64url para ser seguro em URLs e paths
                return self::base64urlEncode($nonce . $encrypted);
            } else {
                // Fallback para OpenSSL com AES-256-GCM (AEAD - autentica e encripta)
                $cipher = 'aes-256-gcm';
                
                // Verifica se GCM está disponível
                if (!in_array($cipher, openssl_get_cipher_methods())) {
                    // Fallback para CBC se GCM não estiver disponível (menos seguro, mas funcional)
                    $cipher = 'AES-256-CBC';
                }
                
                $ivlen = openssl_cipher_iv_length($cipher);
                if ($ivlen === false) {
                    throw new Exception('Invalid cipher');
                }
                
                $iv = openssl_random_pseudo_bytes($ivlen);
                $key = hash('sha256', $sessionKeyBytes, true);
                
                // Para GCM, precisamos de um tag de autenticação
                $tag = '';
                $tagLength = 16; // 128 bits para GCM
                
                if ($cipher === 'aes-256-gcm') {
                    $encrypted = openssl_encrypt($idString, $cipher, $key, OPENSSL_RAW_DATA, $iv, $tag, '', $tagLength);
                } else {
                    // CBC fallback
                    $encrypted = openssl_encrypt($idString, $cipher, $key, OPENSSL_RAW_DATA, $iv);
                }
                
                if ($encrypted === false) {
                    throw new Exception('OpenSSL encryption failed');
                }
                
                // Para GCM, anexa o tag; para CBC, apenas IV + encrypted
                $data = $cipher === 'aes-256-gcm' ? ($iv . $tag . $encrypted) : ($iv . $encrypted);
                
                return self::base64urlEncode($data);
            }
        } catch (Exception $e) {
            throw new Exception('ID encryption error: ' . $e->getMessage());
        }
    }
    
    /**
     * Desencripta um ID usando a chave da sessão
     * 
     * @param string $encryptedId ID encriptado em base64url
     * @return int ID desencriptado como inteiro
     * @throws Exception Se a desencriptação falhar ou o ID não for numérico válido
     */
    public static function decryptId($encryptedId)
    {
        if (empty($encryptedId)) {
            throw new Exception('Empty encrypted ID');
        }
        
        try {
            // Obtém a chave da sessão (já em bytes)
            $sessionKeyBytes = self::getSessionKey();
            
            // Decodifica o base64url
            $decoded = self::base64urlDecode($encryptedId);
            
            if ($decoded === false) {
                throw new Exception('Invalid base64url data');
            }
            
            // Verifica se a extensão Sodium está disponível
            if (function_exists('sodium_crypto_secretbox_open')) {
                // Verifica se tem tamanho mínimo
                if (strlen($decoded) < SODIUM_CRYPTO_SECRETBOX_NONCEBYTES) {
                    throw new Exception('Invalid encrypted data');
                }
                
                // Deriva a mesma chave
                $key = sodium_crypto_generichash($sessionKeyBytes, '', SODIUM_CRYPTO_SECRETBOX_KEYBYTES);
                
                // Separa o nonce dos dados criptografados
                $nonce = substr($decoded, 0, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
                $ciphertext = substr($decoded, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
                
                $decrypted = sodium_crypto_secretbox_open($ciphertext, $nonce, $key);
                
                if ($decrypted === false) {
                    throw new Exception('Sodium decryption failed - invalid key or corrupted data');
                }
            } else {
                // Fallback para OpenSSL
                // Tenta GCM primeiro, depois CBC
                $cipher = 'aes-256-gcm';
                $useGCM = in_array($cipher, openssl_get_cipher_methods());
                
                if (!$useGCM) {
                    $cipher = 'AES-256-CBC';
                }
                
                $ivlen = openssl_cipher_iv_length($cipher);
                
                if ($ivlen === false || strlen($decoded) < $ivlen) {
                    throw new Exception('Invalid encrypted data');
                }
                
                // Separa o IV dos dados
                $iv = substr($decoded, 0, $ivlen);
                $key = hash('sha256', $sessionKeyBytes, true);
                
                if ($useGCM) {
                    // Para GCM, o tag vem após o IV (16 bytes)
                    $tagLength = 16;
                    if (strlen($decoded) < ($ivlen + $tagLength)) {
                        throw new Exception('Invalid encrypted data - missing tag');
                    }
                    $tag = substr($decoded, $ivlen, $tagLength);
                    $ciphertext = substr($decoded, $ivlen + $tagLength);
                    
                    $decrypted = openssl_decrypt($ciphertext, $cipher, $key, OPENSSL_RAW_DATA, $iv, $tag);
                } else {
                    // CBC fallback
                    $ciphertext = substr($decoded, $ivlen);
                    $decrypted = openssl_decrypt($ciphertext, $cipher, $key, OPENSSL_RAW_DATA, $iv);
                }
                
                if ($decrypted === false) {
                    throw new Exception('OpenSSL decryption failed - invalid key or corrupted data');
                }
            }
            
            // Valida que o ID desencriptado é numérico válido (proteção contra injection)
            if (!ctype_digit($decrypted) && !is_numeric($decrypted)) {
                throw new Exception('Decrypted ID is not a valid numeric value');
            }
            
            // Converte para inteiro e retorna
            $id = (int)$decrypted;
            
            // Valida que o ID é positivo (IDs geralmente são > 0)
            if ($id < 0) {
                throw new Exception('Decrypted ID is negative');
            }
            
            return $id;
        } catch (Exception $e) {
            throw new Exception('ID decryption error: ' . $e->getMessage());
        }
    }
    
    /**
     * Limpa a chave de criptografia da sessão
     * Útil para logout ou regeneração de sessão
     */
    public static function clearSessionKey()
    {
        TSession::delValue(self::$sessionKeyName);
    }
    
    /**
     * Verifica se um valor é um ID encriptado
     * 
     * @param string $value Valor a verificar
     * @return bool True se parece ser um ID encriptado
     */
    public static function isEncrypted($value)
    {
        if (empty($value) || !is_string($value)) {
            return false;
        }
        
        // IDs encriptados são base64url, então verificamos se decodifica corretamente
        $decoded = self::base64urlDecode($value);
        return $decoded !== false && strlen($decoded) > 16; // Tamanho mínimo esperado
    }
}

